"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenSearchRoutesService = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _configSchema = require("@osd/config-schema");
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
var _common = require("../../common");
var _helpers = require("./helpers");
var _helpers2 = require("../utils/helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * Server-side routes to process OpenSearch-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerOpenSearchRoutes(router, opensearchRoutesService) {
  router.get({
    path: `${_common.CAT_INDICES_NODE_API_PATH}/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/catIndices/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.GET_MAPPINGS_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/mappings/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.GET_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIndex/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.put({
    path: `${_common.INGEST_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/ingest/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.post({
    path: `${_common.BULK_NODE_API_PATH}/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.BULK_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.SIMULATE_PIPELINE_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.SIMULATE_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      body: _configSchema.schema.object({
        docs: _configSchema.schema.any()
      }),
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string(),
        pipeline_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.get({
    path: `${_common.INGEST_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIngestPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.SEARCH_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getSearchPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
}
class OpenSearchRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "catIndices", async (context, req, res) => {
      const {
        pattern
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('cat.indices', {
          index: pattern,
          format: 'json',
          h: 'health,index'
        });

        // re-formatting the index results to match Index
        const cleanedIndices = response.map(index => ({
          name: index.index,
          health: index.health
        }));
        return res.ok({
          body: cleanedIndices
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getMappings", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        var _response$index;
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.getMapping', {
          index
        });

        // Response will be a dict with key being the index name. Attempt to
        // pull out the mappings. If any errors found (missing index, etc.), an error
        // will be thrown.
        const mappings = (_response$index = response[index]) === null || _response$index === void 0 ? void 0 : _response$index.mappings;
        return res.ok({
          body: mappings
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIndex", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.get', {
          index
        });
        // re-formatting the results to match IndexResponse
        const cleanedIndexDetails = Object.entries(response).map(([indexName, indexDetails]) => ({
          indexName,
          indexDetails
        }));
        return res.ok({
          body: cleanedIndexDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "searchIndex", async (context, req, res) => {
      const {
        index,
        search_pipeline
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      const {
        verbose = false,
        data_source_version = undefined
      } = req.query;
      const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        let response;
        // If verbose is false/undefined, or the version isn't eligible, omit the verbose param when searching.
        if (!verbose || isPreV219) {
          response = await callWithRequest('search', {
            index,
            body,
            search_pipeline
          });
        } else {
          response = await callWithRequest('search', {
            index,
            body,
            search_pipeline,
            verbose_pipeline: verbose
          });
        }
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "ingest", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        index
      } = req.params;
      const doc = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('index', {
          index,
          body: doc
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "bulk", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        pipeline
      } = req.params;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('bulk', {
          body,
          pipeline
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "simulatePipeline", async (context, req, res) => {
      const {
        data_source_id = '',
        pipeline_id = ''
      } = req.params;
      const {
        pipeline,
        docs
      } = req.body;
      const {
        verbose = false
      } = req.query;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        let response = undefined;
        if (!(0, _lodash.isEmpty)(pipeline_id)) {
          response = await callWithRequest('ingest.simulate', {
            body: {
              docs
            },
            id: pipeline_id,
            verbose
          });
        } else {
          response = await callWithRequest('ingest.simulate', {
            body: {
              docs,
              pipeline
            },
            verbose
          });
        }
        return res.ok({
          body: {
            docs: response.docs
          }
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIngestPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('ingest.getPipeline', {
          id: pipeline_id
        });
        // re-formatting the results to match IngestPipelineResponse
        const cleanedIngestPipelineDetails = Object.entries(response).map(([pipelineId, ingestPipelineDetails]) => ({
          pipelineId,
          ingestPipelineDetails
        }));
        return res.ok({
          body: cleanedIngestPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getSearchPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('coreClient.getSearchPipeline', {
          pipeline_id: pipeline_id
        });

        // re-formatting the results to match SearchPipelineResponse
        const cleanedSearchPipelineDetails = Object.entries(response).map(([pipelineId, searchPipelineDetails]) => ({
          pipelineId,
          searchPipelineDetails
        }));
        return res.ok({
          body: cleanedSearchPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.OpenSearchRoutesService = OpenSearchRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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