/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.group;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.GroupIdNotFoundException;
import org.apache.kafka.common.errors.StaleMemberEpochException;
import org.apache.kafka.common.message.OffsetCommitRequestData;
import org.apache.kafka.common.message.OffsetCommitResponseData;
import org.apache.kafka.common.message.OffsetDeleteRequestData;
import org.apache.kafka.common.message.OffsetDeleteResponseData;
import org.apache.kafka.common.message.OffsetFetchRequestData;
import org.apache.kafka.common.message.OffsetFetchResponseData;
import org.apache.kafka.common.message.TxnOffsetCommitRequestData;
import org.apache.kafka.common.message.TxnOffsetCommitResponseData;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.RequestContext;
import org.apache.kafka.common.requests.TransactionResult;
import org.apache.kafka.common.utils.ImplicitLinkedHashCollection;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.coordinator.group.Group;
import org.apache.kafka.coordinator.group.GroupCoordinatorConfig;
import org.apache.kafka.coordinator.group.GroupMetadataManager;
import org.apache.kafka.coordinator.group.OffsetAndMetadata;
import org.apache.kafka.coordinator.group.OffsetExpirationCondition;
import org.apache.kafka.coordinator.group.Record;
import org.apache.kafka.coordinator.group.RecordHelpers;
import org.apache.kafka.coordinator.group.classic.ClassicGroup;
import org.apache.kafka.coordinator.group.classic.ClassicGroupState;
import org.apache.kafka.coordinator.group.generated.OffsetCommitKey;
import org.apache.kafka.coordinator.group.generated.OffsetCommitValue;
import org.apache.kafka.coordinator.group.metrics.GroupCoordinatorMetricsShard;
import org.apache.kafka.coordinator.group.runtime.CoordinatorResult;
import org.apache.kafka.image.MetadataDelta;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.slf4j.Logger;

public class OffsetMetadataManager {
    private final Logger log;
    private final SnapshotRegistry snapshotRegistry;
    private final Time time;
    private MetadataImage metadataImage;
    private final GroupMetadataManager groupMetadataManager;
    private final GroupCoordinatorMetricsShard metrics;
    private final GroupCoordinatorConfig config;
    private final Offsets offsets;
    private final TimelineHashMap<Long, Offsets> pendingTransactionalOffsets;

    OffsetMetadataManager(SnapshotRegistry snapshotRegistry, LogContext logContext, Time time, MetadataImage metadataImage, GroupMetadataManager groupMetadataManager, GroupCoordinatorConfig config, GroupCoordinatorMetricsShard metrics) {
        this.snapshotRegistry = snapshotRegistry;
        this.log = logContext.logger(OffsetMetadataManager.class);
        this.time = time;
        this.metadataImage = metadataImage;
        this.groupMetadataManager = groupMetadataManager;
        this.config = config;
        this.metrics = metrics;
        this.offsets = new Offsets();
        this.pendingTransactionalOffsets = new TimelineHashMap(snapshotRegistry, 0);
    }

    private Group validateOffsetCommit(RequestContext context, OffsetCommitRequestData request) throws ApiException {
        Group group;
        try {
            group = this.groupMetadataManager.group(request.groupId());
        }
        catch (GroupIdNotFoundException ex) {
            if (request.generationIdOrMemberEpoch() < 0) {
                group = this.groupMetadataManager.getOrMaybeCreateClassicGroup(request.groupId(), true);
            }
            if (context.header.apiVersion() >= 9) {
                throw ex;
            }
            throw Errors.ILLEGAL_GENERATION.exception();
        }
        try {
            group.validateOffsetCommit(request.memberId(), request.groupInstanceId(), request.generationIdOrMemberEpoch(), false);
        }
        catch (StaleMemberEpochException ex) {
            if (context.header.apiVersion() >= 9) {
                throw ex;
            }
            throw Errors.UNSUPPORTED_VERSION.exception();
        }
        return group;
    }

    private Group validateTransactionalOffsetCommit(TxnOffsetCommitRequestData request) throws ApiException {
        Group group;
        try {
            group = this.groupMetadataManager.group(request.groupId());
        }
        catch (GroupIdNotFoundException ex) {
            if (request.generationId() < 0) {
                group = this.groupMetadataManager.getOrMaybeCreateClassicGroup(request.groupId(), true);
            }
            throw Errors.ILLEGAL_GENERATION.exception();
        }
        try {
            group.validateOffsetCommit(request.memberId(), request.groupInstanceId(), request.generationId(), true);
        }
        catch (StaleMemberEpochException ex) {
            throw Errors.ILLEGAL_GENERATION.exception();
        }
        return group;
    }

    private void validateOffsetFetch(OffsetFetchRequestData.OffsetFetchRequestGroup request, long lastCommittedOffset) throws GroupIdNotFoundException {
        Group group = this.groupMetadataManager.group(request.groupId(), lastCommittedOffset);
        group.validateOffsetFetch(request.memberId(), request.memberEpoch(), lastCommittedOffset);
    }

    private Group validateOffsetDelete(OffsetDeleteRequestData request) throws GroupIdNotFoundException {
        Group group = this.groupMetadataManager.group(request.groupId());
        group.validateOffsetDelete();
        return group;
    }

    private boolean isMetadataInvalid(String metadata) {
        return metadata != null && metadata.length() > this.config.offsetMetadataMaxSize;
    }

    private static OptionalLong expireTimestampMs(long retentionTimeMs, long currentTimeMs) {
        return retentionTimeMs == -1L ? OptionalLong.empty() : OptionalLong.of(currentTimeMs + retentionTimeMs);
    }

    public CoordinatorResult<OffsetCommitResponseData, Record> commitOffset(RequestContext context, OffsetCommitRequestData request) throws ApiException {
        ClassicGroup classicGroup;
        Group group = this.validateOffsetCommit(context, request);
        if (group.type() == Group.GroupType.CLASSIC && ((classicGroup = (ClassicGroup)group).isInState(ClassicGroupState.STABLE) || classicGroup.isInState(ClassicGroupState.PREPARING_REBALANCE))) {
            this.groupMetadataManager.rescheduleClassicGroupMemberHeartbeat(classicGroup, classicGroup.member(request.memberId()));
        }
        OffsetCommitResponseData response = new OffsetCommitResponseData();
        ArrayList records = new ArrayList();
        long currentTimeMs = this.time.milliseconds();
        OptionalLong expireTimestampMs = OffsetMetadataManager.expireTimestampMs(request.retentionTimeMs(), currentTimeMs);
        request.topics().forEach(topic -> {
            OffsetCommitResponseData.OffsetCommitResponseTopic topicResponse = new OffsetCommitResponseData.OffsetCommitResponseTopic().setName(topic.name());
            response.topics().add(topicResponse);
            topic.partitions().forEach(partition -> {
                if (this.isMetadataInvalid(partition.committedMetadata())) {
                    topicResponse.partitions().add(new OffsetCommitResponseData.OffsetCommitResponsePartition().setPartitionIndex(partition.partitionIndex()).setErrorCode(Errors.OFFSET_METADATA_TOO_LARGE.code()));
                } else {
                    this.log.debug("[GroupId {}] Committing offsets {} for partition {}-{} from member {} with leader epoch {}.", new Object[]{request.groupId(), partition.committedOffset(), topic.name(), partition.partitionIndex(), request.memberId(), partition.committedLeaderEpoch()});
                    topicResponse.partitions().add(new OffsetCommitResponseData.OffsetCommitResponsePartition().setPartitionIndex(partition.partitionIndex()).setErrorCode(Errors.NONE.code()));
                    OffsetAndMetadata offsetAndMetadata = OffsetAndMetadata.fromRequest(partition, currentTimeMs, expireTimestampMs);
                    records.add(RecordHelpers.newOffsetCommitRecord(request.groupId(), topic.name(), partition.partitionIndex(), offsetAndMetadata, this.metadataImage.features().metadataVersion()));
                }
            });
        });
        if (!records.isEmpty()) {
            this.metrics.record("OffsetCommits", records.size());
        }
        return new CoordinatorResult<OffsetCommitResponseData, Record>(records, response);
    }

    public CoordinatorResult<TxnOffsetCommitResponseData, Record> commitTransactionalOffset(RequestContext context, TxnOffsetCommitRequestData request) throws ApiException {
        this.validateTransactionalOffsetCommit(request);
        TxnOffsetCommitResponseData response = new TxnOffsetCommitResponseData();
        ArrayList records = new ArrayList();
        long currentTimeMs = this.time.milliseconds();
        request.topics().forEach(topic -> {
            TxnOffsetCommitResponseData.TxnOffsetCommitResponseTopic topicResponse = new TxnOffsetCommitResponseData.TxnOffsetCommitResponseTopic().setName(topic.name());
            response.topics().add(topicResponse);
            topic.partitions().forEach(partition -> {
                if (this.isMetadataInvalid(partition.committedMetadata())) {
                    topicResponse.partitions().add(new TxnOffsetCommitResponseData.TxnOffsetCommitResponsePartition().setPartitionIndex(partition.partitionIndex()).setErrorCode(Errors.OFFSET_METADATA_TOO_LARGE.code()));
                } else {
                    this.log.debug("[GroupId {}] Committing transactional offsets {} for partition {}-{} from member {} with leader epoch {}.", new Object[]{request.groupId(), partition.committedOffset(), topic.name(), partition.partitionIndex(), request.memberId(), partition.committedLeaderEpoch()});
                    topicResponse.partitions().add(new TxnOffsetCommitResponseData.TxnOffsetCommitResponsePartition().setPartitionIndex(partition.partitionIndex()).setErrorCode(Errors.NONE.code()));
                    OffsetAndMetadata offsetAndMetadata = OffsetAndMetadata.fromRequest(partition, currentTimeMs);
                    records.add(RecordHelpers.newOffsetCommitRecord(request.groupId(), topic.name(), partition.partitionIndex(), offsetAndMetadata, this.metadataImage.features().metadataVersion()));
                }
            });
        });
        if (!records.isEmpty()) {
            this.metrics.record("OffsetCommits", records.size());
        }
        return new CoordinatorResult<TxnOffsetCommitResponseData, Record>(records, response);
    }

    public CoordinatorResult<OffsetDeleteResponseData, Record> deleteOffsets(OffsetDeleteRequestData request) throws ApiException {
        Group group = this.validateOffsetDelete(request);
        ArrayList records = new ArrayList();
        OffsetDeleteResponseData.OffsetDeleteResponseTopicCollection responseTopicCollection = new OffsetDeleteResponseData.OffsetDeleteResponseTopicCollection();
        TimelineHashMap offsetsByTopic = (TimelineHashMap)this.offsets.offsetsByGroup.get((Object)request.groupId());
        request.topics().forEach(topic -> {
            OffsetDeleteResponseData.OffsetDeleteResponsePartitionCollection responsePartitionCollection = new OffsetDeleteResponseData.OffsetDeleteResponsePartitionCollection();
            if (group.isSubscribedToTopic(topic.name())) {
                topic.partitions().forEach(partition -> responsePartitionCollection.add((ImplicitLinkedHashCollection.Element)new OffsetDeleteResponseData.OffsetDeleteResponsePartition().setPartitionIndex(partition.partitionIndex()).setErrorCode(Errors.GROUP_SUBSCRIBED_TO_TOPIC.code())));
            } else {
                TimelineHashMap offsetsByPartition;
                TimelineHashMap timelineHashMap = offsetsByPartition = offsetsByTopic == null ? null : (TimelineHashMap)offsetsByTopic.get((Object)topic.name());
                if (offsetsByPartition != null) {
                    topic.partitions().forEach(partition -> {
                        if (offsetsByPartition.containsKey((Object)partition.partitionIndex())) {
                            responsePartitionCollection.add((ImplicitLinkedHashCollection.Element)new OffsetDeleteResponseData.OffsetDeleteResponsePartition().setPartitionIndex(partition.partitionIndex()));
                            records.add(RecordHelpers.newOffsetCommitTombstoneRecord(request.groupId(), topic.name(), partition.partitionIndex()));
                        }
                    });
                }
            }
            responseTopicCollection.add((ImplicitLinkedHashCollection.Element)new OffsetDeleteResponseData.OffsetDeleteResponseTopic().setName(topic.name()).setPartitions(responsePartitionCollection));
        });
        this.metrics.record("OffsetDeletions", records.size());
        return new CoordinatorResult<OffsetDeleteResponseData, Record>(records, new OffsetDeleteResponseData().setTopics(responseTopicCollection));
    }

    public int deleteAllOffsets(String groupId, List<Record> records) {
        TimelineHashMap offsetsByTopic = (TimelineHashMap)this.offsets.offsetsByGroup.get((Object)groupId);
        AtomicInteger numDeletedOffsets = new AtomicInteger();
        if (offsetsByTopic != null) {
            offsetsByTopic.forEach((topic, offsetsByPartition) -> offsetsByPartition.keySet().forEach(partition -> {
                records.add(RecordHelpers.newOffsetCommitTombstoneRecord(groupId, topic, partition));
                numDeletedOffsets.getAndIncrement();
            }));
        }
        return numDeletedOffsets.get();
    }

    public OffsetFetchResponseData.OffsetFetchResponseGroup fetchOffsets(OffsetFetchRequestData.OffsetFetchRequestGroup request, long lastCommittedOffset) throws ApiException {
        boolean failAllPartitions = false;
        try {
            this.validateOffsetFetch(request, lastCommittedOffset);
        }
        catch (GroupIdNotFoundException ex) {
            failAllPartitions = true;
        }
        ArrayList topicResponses = new ArrayList(request.topics().size());
        TimelineHashMap groupOffsets = failAllPartitions ? null : (TimelineHashMap)this.offsets.offsetsByGroup.get((Object)request.groupId(), lastCommittedOffset);
        request.topics().forEach(topic -> {
            OffsetFetchResponseData.OffsetFetchResponseTopics topicResponse = new OffsetFetchResponseData.OffsetFetchResponseTopics().setName(topic.name());
            topicResponses.add(topicResponse);
            TimelineHashMap topicOffsets = groupOffsets == null ? null : (TimelineHashMap)groupOffsets.get((Object)topic.name(), lastCommittedOffset);
            topic.partitionIndexes().forEach(partitionIndex -> {
                OffsetAndMetadata offsetAndMetadata;
                OffsetAndMetadata offsetAndMetadata2 = offsetAndMetadata = topicOffsets == null ? null : (OffsetAndMetadata)topicOffsets.get(partitionIndex, lastCommittedOffset);
                if (offsetAndMetadata == null) {
                    topicResponse.partitions().add(new OffsetFetchResponseData.OffsetFetchResponsePartitions().setPartitionIndex(partitionIndex.intValue()).setCommittedOffset(-1L).setCommittedLeaderEpoch(-1).setMetadata(""));
                } else {
                    topicResponse.partitions().add(new OffsetFetchResponseData.OffsetFetchResponsePartitions().setPartitionIndex(partitionIndex.intValue()).setCommittedOffset(offsetAndMetadata.offset).setCommittedLeaderEpoch(offsetAndMetadata.leaderEpoch.orElse(-1)).setMetadata(offsetAndMetadata.metadata));
                }
            });
        });
        return new OffsetFetchResponseData.OffsetFetchResponseGroup().setGroupId(request.groupId()).setTopics(topicResponses);
    }

    public OffsetFetchResponseData.OffsetFetchResponseGroup fetchAllOffsets(OffsetFetchRequestData.OffsetFetchRequestGroup request, long lastCommittedOffset) throws ApiException {
        try {
            this.validateOffsetFetch(request, lastCommittedOffset);
        }
        catch (GroupIdNotFoundException ex) {
            return new OffsetFetchResponseData.OffsetFetchResponseGroup().setGroupId(request.groupId()).setTopics(Collections.emptyList());
        }
        ArrayList topicResponses = new ArrayList();
        TimelineHashMap groupOffsets = (TimelineHashMap)this.offsets.offsetsByGroup.get((Object)request.groupId(), lastCommittedOffset);
        if (groupOffsets != null) {
            groupOffsets.entrySet(lastCommittedOffset).forEach(topicEntry -> {
                String topic = (String)topicEntry.getKey();
                TimelineHashMap topicOffsets = (TimelineHashMap)topicEntry.getValue();
                OffsetFetchResponseData.OffsetFetchResponseTopics topicResponse = new OffsetFetchResponseData.OffsetFetchResponseTopics().setName(topic);
                topicResponses.add(topicResponse);
                topicOffsets.entrySet(lastCommittedOffset).forEach(partitionEntry -> {
                    int partition = (Integer)partitionEntry.getKey();
                    OffsetAndMetadata offsetAndMetadata = (OffsetAndMetadata)partitionEntry.getValue();
                    topicResponse.partitions().add(new OffsetFetchResponseData.OffsetFetchResponsePartitions().setPartitionIndex(partition).setCommittedOffset(offsetAndMetadata.offset).setCommittedLeaderEpoch(offsetAndMetadata.leaderEpoch.orElse(-1)).setMetadata(offsetAndMetadata.metadata));
                });
            });
        }
        return new OffsetFetchResponseData.OffsetFetchResponseGroup().setGroupId(request.groupId()).setTopics(topicResponses);
    }

    public boolean cleanupExpiredOffsets(String groupId, List<Record> records) {
        TimelineHashMap offsetsByTopic = (TimelineHashMap)this.offsets.offsetsByGroup.get((Object)groupId);
        if (offsetsByTopic == null) {
            return true;
        }
        Group group = this.groupMetadataManager.group(groupId);
        HashSet expiredPartitions = new HashSet();
        long currentTimestampMs = this.time.milliseconds();
        Optional<OffsetExpirationCondition> offsetExpirationCondition = group.offsetExpirationCondition();
        if (!offsetExpirationCondition.isPresent()) {
            return false;
        }
        AtomicBoolean allOffsetsExpired = new AtomicBoolean(true);
        OffsetExpirationCondition condition = offsetExpirationCondition.get();
        offsetsByTopic.forEach((topic, partitions) -> {
            if (!group.isSubscribedToTopic((String)topic)) {
                partitions.forEach((partition, offsetAndMetadata) -> {
                    if (condition.isOffsetExpired((OffsetAndMetadata)offsetAndMetadata, currentTimestampMs, this.config.offsetsRetentionMs)) {
                        expiredPartitions.add(this.appendOffsetCommitTombstone(groupId, (String)topic, (int)partition, records).toString());
                        this.log.debug("[GroupId {}] Expired offset for partition={}-{}", new Object[]{groupId, topic, partition});
                    } else {
                        allOffsetsExpired.set(false);
                    }
                });
            } else {
                allOffsetsExpired.set(false);
            }
        });
        this.metrics.record("OffsetExpired", expiredPartitions.size());
        return allOffsetsExpired.get();
    }

    private TopicPartition appendOffsetCommitTombstone(String groupId, String topic, int partition, List<Record> records) {
        records.add(RecordHelpers.newOffsetCommitTombstoneRecord(groupId, topic, partition));
        TopicPartition tp = new TopicPartition(topic, partition);
        this.log.trace("[GroupId {}] Removing expired offset and metadata for {}", (Object)groupId, (Object)tp);
        return tp;
    }

    public void replay(long producerId, OffsetCommitKey key, OffsetCommitValue value) {
        String groupId = key.group();
        String topic = key.topic();
        int partition = key.partition();
        if (value != null) {
            try {
                this.groupMetadataManager.group(groupId);
            }
            catch (GroupIdNotFoundException ex) {
                this.groupMetadataManager.getOrMaybeCreateClassicGroup(groupId, true);
            }
            if (producerId == -1L) {
                this.log.debug("Replaying offset commit with key {}, value {}", (Object)key, (Object)value);
                OffsetAndMetadata previousValue = this.offsets.put(groupId, topic, partition, OffsetAndMetadata.fromRecord(value));
                if (previousValue == null) {
                    this.metrics.incrementNumOffsets();
                }
            } else {
                this.log.debug("Replaying transactional offset commit with producer id {}, key {}, value {}", new Object[]{producerId, key, value});
                Offsets pendingOffsets = (Offsets)this.pendingTransactionalOffsets.computeIfAbsent((Object)producerId, __ -> new Offsets());
                pendingOffsets.put(groupId, topic, partition, OffsetAndMetadata.fromRecord(value));
            }
        } else if (this.offsets.remove(groupId, topic, partition) != null) {
            this.metrics.decrementNumOffsets();
        }
    }

    public void replayEndTransactionMarker(long producerId, TransactionResult result) throws RuntimeException {
        Offsets pendingOffsets = (Offsets)this.pendingTransactionalOffsets.remove((Object)producerId);
        if (result == TransactionResult.COMMIT) {
            this.log.debug("Committed transactional offset commits for producer id {}.", (Object)producerId);
            if (pendingOffsets == null) {
                return;
            }
            pendingOffsets.offsetsByGroup.forEach((groupId, topicOffsets) -> topicOffsets.forEach((topicName, partitionOffsets) -> partitionOffsets.forEach((partitionId, offsetAndMetadata) -> {
                this.log.debug("Committed transaction offset commit for producer id {} in group {} with topic {}, partition {}, and offset {}.", new Object[]{producerId, groupId, topicName, partitionId, offsetAndMetadata});
                this.offsets.put(groupId, topicName, partitionId, offsetAndMetadata);
            })));
        } else {
            this.log.debug("Aborted transactional offset commits for producer id {}.", (Object)producerId);
        }
    }

    public void onNewMetadataImage(MetadataImage newImage, MetadataDelta delta) {
        this.metadataImage = newImage;
    }

    OffsetAndMetadata offset(String groupId, String topic, int partition) {
        return this.offsets.get(groupId, topic, partition);
    }

    OffsetAndMetadata pendingTransactionalOffset(long producerId, String groupId, String topic, int partition) {
        Offsets offsets = (Offsets)this.pendingTransactionalOffsets.get((Object)producerId);
        if (offsets == null) {
            return null;
        }
        return offsets.get(groupId, topic, partition);
    }

    private class Offsets {
        private final TimelineHashMap<String, TimelineHashMap<String, TimelineHashMap<Integer, OffsetAndMetadata>>> offsetsByGroup;

        private Offsets() {
            this.offsetsByGroup = new TimelineHashMap(OffsetMetadataManager.this.snapshotRegistry, 0);
        }

        private OffsetAndMetadata get(String groupId, String topic, int partition) {
            TimelineHashMap topicOffsets = (TimelineHashMap)this.offsetsByGroup.get((Object)groupId);
            if (topicOffsets == null) {
                return null;
            }
            TimelineHashMap partitionOffsets = (TimelineHashMap)topicOffsets.get((Object)topic);
            if (partitionOffsets == null) {
                return null;
            }
            return (OffsetAndMetadata)partitionOffsets.get((Object)partition);
        }

        private OffsetAndMetadata put(String groupId, String topic, int partition, OffsetAndMetadata offsetAndMetadata) {
            TimelineHashMap topicOffsets = (TimelineHashMap)this.offsetsByGroup.computeIfAbsent((Object)groupId, __ -> new TimelineHashMap(OffsetMetadataManager.this.snapshotRegistry, 0));
            TimelineHashMap partitionOffsets = (TimelineHashMap)topicOffsets.computeIfAbsent((Object)topic, __ -> new TimelineHashMap(OffsetMetadataManager.this.snapshotRegistry, 0));
            return (OffsetAndMetadata)partitionOffsets.put((Object)partition, (Object)offsetAndMetadata);
        }

        private OffsetAndMetadata remove(String groupId, String topic, int partition) {
            TimelineHashMap topicOffsets = (TimelineHashMap)this.offsetsByGroup.get((Object)groupId);
            if (topicOffsets == null) {
                return null;
            }
            TimelineHashMap partitionOffsets = (TimelineHashMap)topicOffsets.get((Object)topic);
            if (partitionOffsets == null) {
                return null;
            }
            OffsetAndMetadata removedValue = (OffsetAndMetadata)partitionOffsets.remove((Object)partition);
            if (partitionOffsets.isEmpty()) {
                topicOffsets.remove((Object)topic);
            }
            if (topicOffsets.isEmpty()) {
                this.offsetsByGroup.remove((Object)groupId);
            }
            return removedValue;
        }
    }

    public static class Builder {
        private LogContext logContext = null;
        private SnapshotRegistry snapshotRegistry = null;
        private Time time = null;
        private GroupMetadataManager groupMetadataManager = null;
        private MetadataImage metadataImage = null;
        private GroupCoordinatorConfig config = null;
        private GroupCoordinatorMetricsShard metrics = null;

        Builder withLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder withSnapshotRegistry(SnapshotRegistry snapshotRegistry) {
            this.snapshotRegistry = snapshotRegistry;
            return this;
        }

        Builder withTime(Time time) {
            this.time = time;
            return this;
        }

        Builder withGroupMetadataManager(GroupMetadataManager groupMetadataManager) {
            this.groupMetadataManager = groupMetadataManager;
            return this;
        }

        Builder withGroupCoordinatorConfig(GroupCoordinatorConfig config) {
            this.config = config;
            return this;
        }

        Builder withMetadataImage(MetadataImage metadataImage) {
            this.metadataImage = metadataImage;
            return this;
        }

        Builder withGroupCoordinatorMetricsShard(GroupCoordinatorMetricsShard metrics) {
            this.metrics = metrics;
            return this;
        }

        public OffsetMetadataManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.snapshotRegistry == null) {
                this.snapshotRegistry = new SnapshotRegistry(this.logContext);
            }
            if (this.metadataImage == null) {
                this.metadataImage = MetadataImage.EMPTY;
            }
            if (this.time == null) {
                this.time = Time.SYSTEM;
            }
            if (this.groupMetadataManager == null) {
                throw new IllegalArgumentException("GroupMetadataManager cannot be null");
            }
            if (this.metrics == null) {
                throw new IllegalArgumentException("GroupCoordinatorMetricsShard cannot be null");
            }
            return new OffsetMetadataManager(this.snapshotRegistry, this.logContext, this.time, this.metadataImage, this.groupMetadataManager, this.config, this.metrics);
        }
    }
}

